from flask import Flask, render_template, request, jsonify, send_file
import os
import time
from gtts import gTTS
from src import helper
from datetime import datetime
import logging
import atexit
import glob

app = Flask(__name__)
app.secret_key = 'studychatbotsecretkey2025'

# Set up logging
logging.basicConfig(level=logging.DEBUG)

# Track user questions and audio files
user_questions = {}
user_audio_files = {}
MAX_QUESTIONS = 15


def cleanup_old_audio_files():
    """Remove audio files older than 1 hour"""
    try:
        audio_dir = os.path.join('static', 'audio')
        if not os.path.exists(audio_dir):
            return
        
        current_time = time.time()
        for filename in os.listdir(audio_dir):
            file_path = os.path.join(audio_dir, filename)
            if os.path.isfile(file_path):
                # Delete files older than 1 hour
                if current_time - os.path.getctime(file_path) > 3600:
                    os.remove(file_path)
                    logging.info(f"Deleted old audio file: {filename}")
    except Exception as e:
        logging.error(f"Error cleaning up audio files: {e}")

def generate_response(question):
    """Generate a response to the user's question"""
    question_lower = question.lower().strip()
        
    # Default responses for unknown questions
    if "how many" in question_lower and "question" in question_lower:
        return f"You can ask up to {MAX_QUESTIONS} questions in this session."
    
    if "hello" in question_lower or "hi" in question_lower:
        return "Hello! I'm your study assistant. How can I help you with your studies today?"
    
    if "thank" in question_lower:
        return "You're welcome! If you have more questions, feel free to ask."
    
    # For questions we don't have specific answers for
    return f"I'm not sure about '{question}'. As a study assistant, I can help with topics like science, math, history, and study techniques. Try asking about something else!"

def text_to_speech(text, filename):
    """Convert text to speech and save as MP3 with error handling"""
    try:
        tts = gTTS(text=text, lang='en', slow=False)
        tts.save(filename)
        return True
    except Exception as e:
        logging.error(f"Text-to-speech conversion failed: {e}")
        return False

@app.before_request
def before_request():
    """Run cleanup before each request (simplified approach)"""
    # Clean up old files occasionally (not on every request to avoid performance issues)
    if int(time.time()) % 300 == 0:  # Run every 5 minutes
        cleanup_old_audio_files()

@app.route('/')
def home():
    return render_template('index.html')

@app.route('/ask', methods=['POST'])
def ask_question():
    user_id = request.remote_addr  # Use IP as user identifier
    current_time = time.time()
    
    # Initialize user question tracking
    if user_id not in user_questions:
        user_questions[user_id] = {'count': 0, 'last_reset': current_time}
        user_audio_files[user_id] = []
    
    # Reset question count if it's been more than 1 hour
    if current_time - user_questions[user_id]['last_reset'] > 3600:
        user_questions[user_id] = {'count': 0, 'last_reset': current_time}
        # Remove old audio files for this user
        for filename in user_audio_files.get(user_id, []):
            try:
                file_path = os.path.join('static', 'audio', filename)
                if os.path.exists(file_path):
                    os.remove(file_path)
            except Exception as e:
                logging.error(f"Error removing audio file {filename}: {e}")
        user_audio_files[user_id] = []
    
    # Check if user has reached the question limit
    if user_questions[user_id]['count'] >= MAX_QUESTIONS:
        return jsonify({
            'response': f"You've reached your limit of {MAX_QUESTIONS} questions. Please wait before asking more.",
            'audio_url': None,
            'question_count': user_questions[user_id]['count'],
            'max_questions': MAX_QUESTIONS,
            'audio_available': False
        })
    
    # Get the question from the user
    question = request.json.get('question', '').strip()
    
    if not question:
        return jsonify({'error': 'No question provided'}), 400
    
    # Generate response
    #response_text = generate_response(question)
    response_text = helper.ask_gemini(question)
    # Generate audio file
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    filename = f"response_{user_id}_{timestamp}.mp3"
    audio_path = os.path.join('static', 'audio', filename)
    
    # Ensure directory exists
    os.makedirs(os.path.dirname(audio_path), exist_ok=True)
    
    # Create audio file
    audio_success = text_to_speech(response_text, audio_path)
    
    # Update question count and track audio file
    user_questions[user_id]['count'] += 1
    if audio_success:
        user_audio_files[user_id].append(filename)
    
    return jsonify({
        'response': response_text,
        'audio_url': f'/static/audio/{filename}' if audio_success else None,
        'question_count': user_questions[user_id]['count'],
        'max_questions': MAX_QUESTIONS,
        'audio_available': audio_success
    })

@app.route('/audio/<filename>')
def serve_audio(filename):
    return send_file(os.path.join('static', 'audio', filename))

@app.route('/cleanup_audio', methods=['POST'])
def cleanup_audio():
    """Clean up audio files for a specific user"""
    user_id = request.remote_addr
    deleted_files = []
    
    if user_id in user_audio_files:
        for filename in user_audio_files[user_id]:
            try:
                file_path = os.path.join('static', 'audio', filename)
                if os.path.exists(file_path):
                    os.remove(file_path)
                    deleted_files.append(filename)
            except Exception as e:
                logging.error(f"Error removing audio file {filename}: {e}")
        
        # Remove user from tracking
        user_audio_files.pop(user_id, None)
        user_questions.pop(user_id, None)
    
    return jsonify({
        'status': 'success',
        'deleted_files': deleted_files,
        'message': f'Deleted {len(deleted_files)} audio files'
    })

# Clean up all audio files on server shutdown
@atexit.register
def shutdown_cleanup():
    """Clean up all audio files when the server shuts down"""
    try:
        audio_dir = os.path.join('static', 'audio')
        if os.path.exists(audio_dir):
            for filename in os.listdir(audio_dir):
                file_path = os.path.join(audio_dir, filename)
                if os.path.isfile(file_path):
                    os.remove(file_path)
            logging.info("Cleaned up all audio files on server shutdown")
    except Exception as e:
        logging.error(f"Error during shutdown cleanup: {e}")

if __name__ == '__main__':
    # Run cleanup once at startup
    cleanup_old_audio_files()
    app.run(debug=True)